package main

import (
	"context"
	"fmt"
	"os"
	"time"

	"mydemo/pkg/fio/client"
	"mydemo/pkg/fio/client/localmarket"
	"mydemo/pkg/fio/models"

	"github.com/gen2brain/beeep"
)

func main() {
	// Validate argument count
	if len(os.Args) != 3 {
		fmt.Println("Must provide a planet name and resource ticker")
		fmt.Printf("Example: %s Katoa SF\n", os.Args[0])
		os.Exit(1)
	}

	// Extract arguments
	planet := os.Args[1]
	material := os.Args[2]

	// Last count of ads seen
	lastBuyCount := -1
	lastSellCount := -1

	// Loop forever
	for {
		// Get the matching buy ads (b), sell ads (s), and any error
		b, s, err := getAds(planet, material)

		if err != nil {
			// Show the error if we hit one
			fmt.Println(err)
		} else {
			currentBuyCount := len(b)
			currentSellCount := len(s)

			if currentBuyCount != lastBuyCount || currentSellCount != lastSellCount {
				// Create message
				message := fmt.Sprintf("%s - %d buy - %d sell", material, currentBuyCount, currentSellCount)

				// Deploy the notification!
				err := beeep.Notify(planet, message, "")
				if err != nil {
					fmt.Println(err)
				}
			}

			lastBuyCount = currentBuyCount
			lastSellCount = currentSellCount
		}

		// Time to sleep
		time.Sleep(1 * time.Minute)
	}
}

func getAds(planet, material string) ([]*models.BuySellAd, []*models.BuySellAd, error) {
	// Use the API client generated by swagger
	fioc := client.Default.Localmarket

	// Get all ads from given planet. GetAdsAt was generated by swagger.
	ok, empty, err := fioc.GetAdsAt(&localmarket.GetAdsAtParams{
		Planet:  planet,
		Context: context.TODO(),
	})

	// Something went wrong calling to FIO
	if err != nil {
		return nil, nil, err
	}

	// No ads at all. Probably a planet name type.
	if empty != nil {
		return nil, nil, fmt.Errorf("No ads at %s (is planet name correct?)\n", planet)
	}

	var matchingBuy []*models.BuySellAd
	var matchingSell []*models.BuySellAd

	// Filter Buy Ads
	for _, ad := range ok.GetPayload().BuyingAds {
		if ad.MaterialTicker == material {
			matchingBuy = append(matchingBuy, ad)
		}
	}

	// Filter Sell Ads
	for _, ad := range ok.GetPayload().SellingAds {
		if ad.MaterialTicker == material {
			matchingSell = append(matchingSell, ad)
		}
	}

	return matchingBuy, matchingSell, nil
}
